#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

#include "util/generic_conf_parser.h"

typedef enum parse_type_t
{
	PARSE_EARLY,
	PARSE_DEFERRED,
	PARSE_ALL
} parse_type_t;

static const kml_conf_parser_t *current_parser=NULL;

static char *conf_file_buf=NULL;
static size_t conf_file_size=0;

static const kml_conf_parser_t **parser_ary;

static error_code_t generic_conf_parser_do_parse(parse_type_t parse_type);

static error_code_t generic_conf_parser_search_begin_of_line(char **start_line_ptr, char *running_ptr);
static error_code_t generic_conf_parser_search_end_of_line(char **start_line_ptr, char *running_ptr,
		parse_type_t parse_type);

static error_code_t generic_conf_parser_process_section_tag(const char *line, parse_type_t parse_type);

#ifdef UNIT_TESTING
error_code_t generic_conf_parser_process_line(char *line, parse_type_t parse_type);
#else
static error_code_t generic_conf_parser_process_line(char *line, parse_type_t parse_type);
#endif

void generic_conf_parser_init(const kml_conf_parser_t **parser)
{
	parser_ary=parser;
	current_parser=0;
}

error_code_t generic_conf_parser_read_file(const char *conf_file)
{
	struct stat stat_result;
	int fd;
	size_t bytes_read_sum=0;
	ssize_t bytes_read;

	if (stat(conf_file,&stat_result)!=0)
	{
		logger_log_error("Configuration file not found: %s",conf_file);
		return RESULT_INVALID;
	}

	conf_file_size=(size_t)stat_result.st_size;
	conf_file_buf=malloc(conf_file_size);
	if (conf_file_buf==NULL)
		return RESULT_NORESOURCES;

	fd=open(conf_file, O_RDONLY);
	if (fd==-1)
	{
		logger_log_error("Unable to read configuration file: %s",strerror(errno));
		return RESULT_INVALID;
	}

	while ((bytes_read=read(fd,conf_file_buf+bytes_read_sum,conf_file_size-bytes_read_sum))>0)
		bytes_read_sum+=(size_t)bytes_read;

	close(fd);
	logger_log_debug("GENERIC_CONF_FILE_PARSER -> Read in configuration file: %s, size: %d",conf_file,conf_file_size);
	return RESULT_OK;
}

char *generic_conf_parser_trimline(char *string, bool *is_empty)
{
	size_t len;
	char* p_begin;
	char* p_end;

	len=strlen(string);

	p_begin=string;
	//end points to the character before the terminating '\0'
	p_end=string+len-1;

	//remove leading spaces
	//each string is terminated by '\0' so this loop will end
	while((*p_begin)==' ')
		p_begin++;

	//remove trailing spaces or new lines
	while(((*p_end)=='\n' || (*p_end)==' ' || (*p_end)=='\t') && p_end >= p_begin)
	{
		(*p_end)='\0';
		p_end--;
	}

	if (is_empty!=NULL)
		*is_empty=(*p_begin)=='\0';

	return p_begin;
}

bool generic_conf_parser_file_read_in(void)
{
	return conf_file_buf!=NULL;
}

error_code_t generic_conf_parser_parse(void)
{
	if (conf_file_buf!=NULL)
	{
		logger_log_debug("GENERIC_CONF_PARSER -> Parsing the complete configuration file.");
		return generic_conf_parser_do_parse(PARSE_ALL);
	}
	else
		return RESULT_INVALID;
}

error_code_t generic_conf_parser_parse_early(void)
{
	if (conf_file_buf!=NULL)
	{
		logger_log_debug("GENERIC_CONF_PARSER -> Parsing early sections.");
		return generic_conf_parser_do_parse(PARSE_EARLY);
	}
	else
		return RESULT_INVALID;
}

error_code_t generic_conf_parser_parse_deferred(void)
{
	if (conf_file_buf!=NULL)
	{
		logger_log_debug("GENERIC_CONF_PARSER -> Parsing deferred sections.");
		return generic_conf_parser_do_parse(PARSE_DEFERRED);
	}
	else
		return RESULT_INVALID;
}

void generic_conf_parser_deinit(void)
{
	if (conf_file_buf!=NULL)
		free(conf_file_buf);
	conf_file_buf=NULL;
	current_parser=NULL;
}

error_code_t generic_conf_parser_add_parameters(char **parameter_list, char *parameters)
{
	if (!*parameter_list)
	{
		*parameter_list = (char *) malloc(sizeof(char));
		if (*parameter_list == NULL)
		{
		        return RESULT_NORESOURCES;
		}
		**parameter_list = '\0';
	}
	ssize_t buf_size = strlen(*parameter_list) + strlen(parameters) + 2 * (sizeof(char));
	char *tmp = realloc(*parameter_list, buf_size);
	if (!tmp)
	{
		return RESULT_NORESOURCES;
	}

	*parameter_list = tmp;
	strncat(*parameter_list, parameters, buf_size - strlen(*parameter_list) - 1);
	strncat(*parameter_list, " ", buf_size - strlen(*parameter_list) - 1);
	return RESULT_OK;
}

static error_code_t generic_conf_parser_do_parse(parse_type_t parse_type)
{
	error_code_t result=RESULT_OK;;
	char *running_ptr=conf_file_buf;
	char *start_line_ptr=NULL;

	//points behind the last valid character in the buffer
	const char *_END_OF_BUF=conf_file_buf+conf_file_size;

	while (running_ptr<_END_OF_BUF && result == RESULT_OK)
	{
		if (start_line_ptr==NULL)
			result=generic_conf_parser_search_begin_of_line(&start_line_ptr, running_ptr);
		else
			result=generic_conf_parser_search_end_of_line(&start_line_ptr, running_ptr, parse_type);

		running_ptr++;
	}

	return result;
}

static error_code_t generic_conf_parser_search_begin_of_line(char **start_line_ptr, char *running_ptr)
{
	//skip leading spaces, tabs, or empty lines
	if (*running_ptr!=' ' && *running_ptr!='\t' && *running_ptr!='\n' && *running_ptr!='\0')
		*start_line_ptr=running_ptr;

	return RESULT_OK;
}

static error_code_t generic_conf_parser_search_end_of_line(char **start_line_ptr, char *running_ptr,
		parse_type_t parse_type)
{
	char *line;
	bool line_is_empty;
	error_code_t result=RESULT_OK;

	// '\n' and '\0' indicate the end of a line. The latter case happens when we are parsing the second time due to the
	// replacement of all '\n' by '\0'
	if (*running_ptr!='\n' && *running_ptr!='\0')
		return RESULT_OK;

	// prepare the line by replacing EOL by '\0'. This way we can pass the line as conventional terminated
	// string to the parsers
	if (*running_ptr=='\n')
		*running_ptr='\0';

	//trim the line
	line=generic_conf_parser_trimline(*start_line_ptr,&line_is_empty);

	//skip empty lines or lines with comments
	if (!line_is_empty && line[0]!='#')
		result=generic_conf_parser_process_line(line, parse_type);

	//reset the start_line_ptr to go on searching the next line
	*start_line_ptr=NULL;

	return result;
}

static error_code_t generic_conf_parser_process_section_tag(const char *line, parse_type_t parse_type)
{
	const kml_conf_parser_t **parser_itr=parser_ary;

	current_parser=*parser_itr;
	while (current_parser!=NULL)
	{
		if ((parse_type==PARSE_EARLY && !current_parser->can_defere_parsing)   ||
			(parse_type==PARSE_DEFERRED && current_parser->can_defere_parsing) ||
			parse_type==PARSE_ALL)
		{
			if (strcmp(line, current_parser->section_name)==0)
			{
				logger_log_debug("GENERIC_CONF_PARSER -> Now parsing section \'%s\' in configuration file.", line);
				//found a matching parser -> select it
				return RESULT_OK;
			}
		}
		parser_itr++;
		current_parser=*parser_itr;
	}
	return RESULT_OK;
}

//export the function for testing
#ifdef UNIT_TESTING
error_code_t generic_conf_parser_process_line(char *line, parse_type_t parse_type)
#else
static error_code_t generic_conf_parser_process_line(char *line, parse_type_t parse_type)
#endif
{
	if (line[0]=='[')
		return generic_conf_parser_process_section_tag(line, parse_type);

	if (current_parser!=NULL)
		return current_parser->parse_line(line);

	return RESULT_OK;
}

